/**
* \file: AilConfiguration.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Michalik / ADIT/SW2 / jmichalik@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <vector>
#include <algorithm>
#include <adit_logging.h>
#include "AilConfiguration.h"

LOG_IMPORT_CONTEXT(cply);

using namespace std;

namespace adit { namespace carplay
{

static const char* _configStrings[AilConfiguration::Channel_Count] =
{
    "alsa-main-audio", "alsa-alternate-audio", "alsa-main-audio-in"
};

static const char* _nameStrings[AilConfiguration::Channel_Count] =
{
    "main audio out", "alternate audio out", "main audio in"
};


static std::map<std::string, std::string> _parseMap(const std::string& inValue, bool inLowerCase,
        const std::string& inLog);

AilConfiguration* AilConfiguration::instance = nullptr;

AilConfiguration& AilConfiguration::Instance()
{
    if (instance == nullptr)
        instance = new AilConfiguration();
    return *instance;
}

AilConfiguration::AilConfiguration()
{
    verbose = -1;
    ailLogging = -1;
}

bool AilConfiguration::IsVerboseLogging(const IConfiguration& inConfig)
{
    if (verbose == -1)
    {
        string verboseConfig;
        if (inConfig.TryGetItem("enable-verbose-logging", verboseConfig))
        {
            verbose = (0 != inConfig.GetNumber("enable-verbose-logging", 0)) ? 1 : 0;
            if(verbose)
                LOG_WARN((cply, "verbose logging enabled, do not use in production!"));
        }
        else
            verbose = 0;
    }

    return verbose == 1;
}

bool AilConfiguration::IsAilLogging(const IConfiguration& inConfig)
{
    if (ailLogging == -1)
    {
        string ailConfig;
        if (inConfig.TryGetItem("alsa-dump-configuration", ailConfig))
        {
            ailLogging = (0 != inConfig.GetNumber("alsa-dump-configuration", 0)) ? 1 : 0;
            LOG_WARN((cply, "ail dump configuration enabled, do not use in production!"));
        }
        else
            ailLogging = 0;
    }

    return ailLogging == 1;
}

bool AilConfiguration::GetDeviceSettings(const IConfiguration& inConfig, Channel inChannel, const string& inAudioType,
        AudioFormatStruct inFormat, string& outDeviceName, uint32_t& outPeriodMilli, int32_t& outBufferPeriods,
        int32_t& outSilenceMilli, int32_t& outInitToutMilli)
{
    if (!parseDeviceSettings(inConfig, inChannel))
    {
        LOG_ERROR((cply, "could not parse audio device settings %d", inChannel));
        return false;
    }

    bool applied = false;

    // go through list and apply if all conditions match
    for (auto setting : deviceSettings[inChannel])
    {
        bool ok = true;
        ok &= (setting.SampleRate == -1 || setting.SampleRate == inFormat.SampleRate);
        ok &= (setting.SampleSize == -1 || setting.SampleSize * 8 == inFormat.BitsPerChannel);
        ok &= (setting.Channels == -1 || setting.Channels == inFormat.Channels);
        ok &= (setting.AudioType.empty() || setting.AudioType.compare(inAudioType) == 0);

        if (ok)
        {
            // apply
            applied = true;
            if (!setting.DeviceName.empty())
                outDeviceName = setting.DeviceName;
            if (setting.PeriodMilli != -1)
                outPeriodMilli = setting.PeriodMilli;
            if (setting.BufferPeriods != -1)
                outBufferPeriods = setting.BufferPeriods;
            if (setting.SilenceMilli != -1)
                outSilenceMilli = setting.SilenceMilli;
            if (setting.InitToutMilli != -1)
                outInitToutMilli = setting.InitToutMilli;

        }
    }

    if (!applied)
        LOG_ERROR((cply, "no audio device setting could be found"));

    return applied;
}

bool AilConfiguration::parseDeviceSettings(const IConfiguration& inConfig, Channel inChannel)
{
    bool reparse = (!deviceSettings[inChannel].empty());

    auto list = inConfig.GetItems(_configStrings[inChannel]);

    LOGD_DEBUG((cply, "%sparse audio device settings for %s", reparse ? "re-" : "",
            _nameStrings[inChannel]));
    for (auto line : list)
    {
        AudioDeviceSettings setting;
        map < string, string > settingsMap = _parseMap(line, true,
                _nameStrings[inChannel] /* used for logging */);

        // note, parsed keys are converted to small letters

        auto iter = settingsMap.find("device");
        if (iter != settingsMap.end())
            setting.DeviceName = iter->second;
        iter = settingsMap.find("periodms");
        if (iter != settingsMap.end())
            setting.PeriodMilli = atol(iter->second.c_str());
        iter = settingsMap.find("bufferperiods");
        if (iter != settingsMap.end())
            setting.BufferPeriods = atol(iter->second.c_str());
        iter = settingsMap.find("silencems");
        if (iter != settingsMap.end())
            setting.SilenceMilli = atol(iter->second.c_str());
        iter = settingsMap.find("audiotype");
        if (iter != settingsMap.end())
            setting.AudioType = iter->second;
        iter = settingsMap.find("samplerate");
        if (iter != settingsMap.end())
            setting.SampleRate = atol(iter->second.c_str());
        iter = settingsMap.find("samplesize");
        if (iter != settingsMap.end())
            setting.SampleSize = atol(iter->second.c_str());
        iter = settingsMap.find("channels");
        if (iter != settingsMap.end())
            setting.Channels = atol(iter->second.c_str());
        iter = settingsMap.find("inittoutms");
        if (iter != settingsMap.end())
            setting.InitToutMilli = atol(iter->second.c_str());

        setting.Dump();

        if (reparse)
        {
            // replace only first item
            deviceSettings[inChannel].erase(deviceSettings[inChannel].begin());
            deviceSettings[inChannel].insert(deviceSettings[inChannel].begin(), setting);
            break;
        }
        else
        {
            // append to list
            deviceSettings[inChannel].push_back(setting);
        }
    }

    return true;
}

AilConfiguration::AudioDeviceSettings::AudioDeviceSettings()
{
    PeriodMilli = -1;
    BufferPeriods = -1;
    SilenceMilli = -1;
    SampleRate = -1;
    SampleSize = -1;
    Channels = -1;
    InitToutMilli = -1;
}

void AilConfiguration::AudioDeviceSettings::Dump()
{
    LOGD_DEBUG((cply, "device=%s periodMs=%d bufferPeriods=%d silenceMs=%d audioType=%s sampleRate=%d sampleSize=%d channels=%d initTout=%d",
            DeviceName.c_str(), PeriodMilli, BufferPeriods, SilenceMilli, AudioType.c_str(), SampleRate, SampleSize,
            Channels, InitToutMilli));
}

static vector<string> _splitString(const string& inValue, const char inSeparators[])
{
    vector<string> result;

    string::size_type current = 0;
    string::size_type found;
    while (string::npos != (found = inValue.find_first_of(inSeparators, current)))
    {
        // only add string to vector if there is no additional whitespace
        if(current != found)
        {
            result.emplace_back(inValue, current, found - current);
        }
        current = found + 1;
    }

    result.emplace_back(inValue, current);
    return result;
}

static string _trimString(const string& inValue)
{
    string::size_type begin = inValue.find_first_not_of(" \t");
    if (begin == string::npos)
        begin = 0;
    string::size_type length = inValue.find_last_not_of(" \t");
    if (length == string::npos)
        length = inValue.length() - 1;
    else
        length = length - begin + 1;
    return inValue.substr(begin, length);
}

map<string, string> _parseMap(const string& inValue, bool inLowerCase, const string& inLog)
{
    map<string, string> dictionary;

    // split the string into 'xx=yy' key-value pairs
    vector<string> keyValuePairs = _splitString(inValue, " \t,");
    for (auto keyValuePair : keyValuePairs)
    {
        // split into key and value
        vector<string> keyValueVector = _splitString(keyValuePair, "=");

        // in case we have multiple or no '='
        if (keyValueVector.size() != 2)
        {
            LOG_ERROR((cply, "%s: could not parse %s as key-value pair",
                    inLog.c_str(), _trimString(keyValuePair).c_str()));
        }
        // in case we have a clear key-value pair
        else
        {
            string key = _trimString(keyValueVector[0]);
            if (inLowerCase)
            {
                std::transform(key.begin(), key.end(), key.begin(),
                        std::ptr_fun<int, int>(std::tolower));
            }

            string value = _trimString(keyValueVector[1]);

            // insert into result map
            dictionary.insert(pair<string, string>(key, value));
        }
    }

    return dictionary;
}

} } // namespace adit { namespace carplay
